/* Proposed SG14 status_code
(C) 2018-2025 Niall Douglas <http://www.nedproductions.biz/> (5 commits)
File Created: Jun 2018


Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#ifndef BOOST_OUTCOME_SYSTEM_ERROR2_ERRORED_STATUS_CODE_HPP
#define BOOST_OUTCOME_SYSTEM_ERROR2_ERRORED_STATUS_CODE_HPP

#include "quick_status_code_from_enum.hpp"

BOOST_OUTCOME_SYSTEM_ERROR2_NAMESPACE_BEGIN

namespace detail
{
  template <class T> inline void is_errored_check_function(T *v)
  {
    if(v->success())
    {
      std::terminate();
    }
  }
}  // namespace detail

/*! A `status_code` which is always a failure. The closest equivalent to
`std::error_code`, except it cannot be modified, and is templated.

Differences from `status_code`:

- Never successful (this contract is checked on construction, if fails then it
terminates the process).
- Is immutable.
*/
template <class DomainType>
class errored_status_code : public detail::status_code_impl<DomainType, detail::is_errored_check_function>
{
  using _base = detail::status_code_impl<DomainType, detail::is_errored_check_function>;
  using _base::clear;
  using _base::success;

public:
  using _base::_base;

  //! The type of the domain.
  using typename _base::domain_type;
  //! The type of the error code.
  using typename _base::value_type;
  //! The type of a reference to a message string.
  using typename _base::string_ref;

  //! Default constructor.
  errored_status_code() = default;
  //! Copy constructor.
  errored_status_code(const errored_status_code &) = default;
  //! Move constructor.
  errored_status_code(errored_status_code &&) = default;  // NOLINT
  //! Copy assignment.
  errored_status_code &operator=(const errored_status_code &) = default;
  //! Move assignment.
  errored_status_code &operator=(errored_status_code &&) = default;  // NOLINT
  ~errored_status_code() = default;

  //! Explicitly construct from any similarly erased status code
  explicit errored_status_code(const _base &o) noexcept(std::is_nothrow_copy_constructible<_base>::value)
      : _base(o)
  {
    detail::is_errored_check_function(this);
  }
  //! Explicitly construct from any similar status code
  explicit errored_status_code(_base &&o) noexcept(std::is_nothrow_move_constructible<_base>::value)
      : _base(static_cast<_base &&>(o))
  {
    detail::is_errored_check_function(this);
  }

  //! Implicit construction from any type where an ADL discovered `make_status_code(T, Args ...)` returns a
  //! `status_code`.
  BOOST_OUTCOME_SYSTEM_ERROR2_TEMPLATE(class T, class... Args,  //
                         class MakeStatusCodeResult = typename detail::safe_get_make_status_code_result<
                         T, Args...>::type)  // Safe ADL lookup of make_status_code(), returns void if not found
  BOOST_OUTCOME_SYSTEM_ERROR2_TREQUIRES(
  BOOST_OUTCOME_SYSTEM_ERROR2_TPRED(!std::is_same<typename std::decay<T>::type, errored_status_code>::value  // not copy/move of self
                      && !std::is_same<typename std::decay<T>::type, in_place_t>::value        // not in_place_t
                      && is_status_code<MakeStatusCodeResult>::value                  // ADL makes a status code
                      && std::is_constructible<_base, MakeStatusCodeResult>::value))  // ADLed status code is compatible
  errored_status_code(T &&v,
                      Args &&...args) noexcept(detail::safe_get_make_status_code_noexcept<T, Args...>::value)  // NOLINT
      : _base(make_status_code(static_cast<T &&>(v), static_cast<Args &&>(args)...))
  {
    detail::is_errored_check_function(this);
  }

  //! Return a copy of the code.
  BOOST_OUTCOME_SYSTEM_ERROR2_CONSTEXPR14 errored_status_code clone() const { return *this; }

  //! Always false (including at compile time), as errored status codes are never successful.
  constexpr bool success() const noexcept { return false; }
  //! Return a const reference to the `value_type`.
  constexpr const value_type &value() const & noexcept { return this->_value; }
};

namespace traits
{
  template <class DomainType> struct is_move_bitcopying<errored_status_code<DomainType>>
  {
    static constexpr bool value = is_move_bitcopying<typename DomainType::value_type>::value;
  };
}  // namespace traits

template <class ErasedType>
class errored_status_code<detail::erased<ErasedType>>
    : public detail::status_code_impl<detail::erased<ErasedType>, detail::is_errored_check_function>
{
  using _base = detail::status_code_impl<detail::erased<ErasedType>, detail::is_errored_check_function>;
  using _base::success;

public:
  using _base::_base;

  using domain_type = typename _base::domain_type;
  using value_type = typename _base::value_type;
  using string_ref = typename _base::string_ref;

  //! Default construction to empty
  errored_status_code() = default;
  //! Copy constructor
  errored_status_code(const errored_status_code &) = default;
  //! Move constructor
  errored_status_code(errored_status_code &&) = default;  // NOLINT
                                                          //! Copy assignment
  errored_status_code &operator=(const errored_status_code &) = default;
  //! Move assignment
  errored_status_code &operator=(errored_status_code &&) = default;  // NOLINT
  ~errored_status_code() = default;

  //! Explicitly construct from any similarly erased status code
  explicit errored_status_code(const _base &o) noexcept(std::is_nothrow_copy_constructible<_base>::value)
      : _base(o)
  {
    detail::is_errored_check_function(this);
  }
  //! Explicitly construct from any similarly erased status code
  explicit errored_status_code(_base &&o) noexcept(std::is_nothrow_move_constructible<_base>::value)
      : _base(static_cast<_base &&>(o))
  {
    detail::is_errored_check_function(this);
  }

  //! Implicit construction from any type where an ADL discovered `make_status_code(T, Args ...)` returns a
  //! `status_code`.
  BOOST_OUTCOME_SYSTEM_ERROR2_TEMPLATE(class T, class... Args,  //
                         class MakeStatusCodeResult = typename detail::safe_get_make_status_code_result<
                         T, Args...>::type)  // Safe ADL lookup of make_status_code(), returns void if not found
  BOOST_OUTCOME_SYSTEM_ERROR2_TREQUIRES(
  BOOST_OUTCOME_SYSTEM_ERROR2_TPRED(!std::is_same<typename std::decay<T>::type, errored_status_code>::value  // not copy/move of self
                      && !std::is_same<typename std::decay<T>::type, value_type>::value  // not copy/move of value type
                      && is_status_code<MakeStatusCodeResult>::value                     // ADL makes a status code
                      && std::is_constructible<_base, MakeStatusCodeResult>::value))  // ADLed status code is compatible
  errored_status_code(T &&v,
                      Args &&...args) noexcept(detail::safe_get_make_status_code_noexcept<T, Args...>::value)  // NOLINT
      : _base(make_status_code(static_cast<T &&>(v), static_cast<Args &&>(args)...))
  {
    detail::is_errored_check_function(this);
  }

  //! Return a copy of the erased code by asking the domain to perform the erased copy.
  BOOST_OUTCOME_SYSTEM_ERROR2_CONSTEXPR20 errored_status_code clone() const
  {
    if(nullptr == this->_domain)
    {
      return {};
    }
    errored_status_code x;
    const int errcode = this->_domain->_do_erased_copy(x, *this, this->_domain->payload_info());
    if(errcode != 0)
    {
      BOOST_OUTCOME_SYSTEM_ERROR2_FATAL("errored_status_code<erased>::clone() failed");
    }
    return x;
  }

  //! Permit casting to `status_code`
  BOOST_OUTCOME_SYSTEM_ERROR2_CONSTEXPR14 operator const status_code<detail::erased<ErasedType>> &() const &
  {
    return static_cast<const status_code<detail::erased<ErasedType>> &>(
    static_cast<const detail::status_code_storage<detail::erased<ErasedType>> &>(*this));
  }
  //! Permit casting to `status_code`
  BOOST_OUTCOME_SYSTEM_ERROR2_CONSTEXPR14 operator status_code<detail::erased<ErasedType>> &&() &&
  {
    return static_cast<status_code<detail::erased<ErasedType>> &&>(
    static_cast<detail::status_code_storage<detail::erased<ErasedType>> &&>(*this));
  }

  //! Always false (including at compile time), as errored status codes are never successful.
  constexpr bool success() const noexcept { return false; }
  //! Return the erased `value_type` by value.
  constexpr value_type value() const noexcept { return this->_value; }
};
/*! An erased type specialisation of `errored_status_code<D>`.
Available only if `ErasedType` satisfies `traits::is_move_bitcopying<ErasedType>::value`.
*/
template <class ErasedType> using erased_errored_status_code = errored_status_code<detail::erased<ErasedType>>;


namespace traits
{
  template <class ErasedType> struct is_move_bitcopying<errored_status_code<detail::erased<ErasedType>>>
  {
    static constexpr bool value = true;
  };
}  // namespace traits


//! True if the status code's are semantically equal via `equivalent()`.
template <class DomainType1, class DomainType2>
inline bool operator==(const errored_status_code<DomainType1> &a, const errored_status_code<DomainType2> &b) noexcept
{
  return a.equivalent(static_cast<const status_code<DomainType2> &>(b));
}
//! True if the status code's are semantically equal via `equivalent()`.
template <class DomainType1, class DomainType2>
inline bool operator==(const status_code<DomainType1> &a, const errored_status_code<DomainType2> &b) noexcept
{
  return a.equivalent(static_cast<const status_code<DomainType2> &>(b));
}
//! True if the status code's are semantically equal via `equivalent()`.
template <class DomainType1, class DomainType2>
inline bool operator==(const errored_status_code<DomainType1> &a, const status_code<DomainType2> &b) noexcept
{
  return static_cast<const status_code<DomainType1> &>(a).equivalent(b);
}
//! True if the status code's are not semantically equal via `equivalent()`.
template <class DomainType1, class DomainType2>
inline bool operator!=(const errored_status_code<DomainType1> &a, const errored_status_code<DomainType2> &b) noexcept
{
  return !a.equivalent(static_cast<const status_code<DomainType2> &>(b));
}
//! True if the status code's are not semantically equal via `equivalent()`.
template <class DomainType1, class DomainType2>
inline bool operator!=(const status_code<DomainType1> &a, const errored_status_code<DomainType2> &b) noexcept
{
  return !a.equivalent(static_cast<const status_code<DomainType2> &>(b));
}
//! True if the status code's are not semantically equal via `equivalent()`.
template <class DomainType1, class DomainType2>
inline bool operator!=(const errored_status_code<DomainType1> &a, const status_code<DomainType2> &b) noexcept
{
  return !static_cast<const status_code<DomainType1> &>(a).equivalent(b);
}
//! True if the status code's are semantically equal via `equivalent()` to `make_status_code(T)`.
BOOST_OUTCOME_SYSTEM_ERROR2_TEMPLATE(class DomainType1, class T,  //
                       class MakeStatusCodeResult = typename detail::safe_get_make_status_code_result<
                       const T &>::type)  // Safe ADL lookup of make_status_code(), returns void if not found
BOOST_OUTCOME_SYSTEM_ERROR2_TREQUIRES(BOOST_OUTCOME_SYSTEM_ERROR2_TPRED(is_status_code<MakeStatusCodeResult>::value))  // ADL makes a status code
inline bool operator==(const errored_status_code<DomainType1> &a, const T &b)
{
  return a.equivalent(make_status_code(b));
}
//! True if the status code's are semantically equal via `equivalent()` to `make_status_code(T)`.
BOOST_OUTCOME_SYSTEM_ERROR2_TEMPLATE(class T, class DomainType1,  //
                       class MakeStatusCodeResult = typename detail::safe_get_make_status_code_result<
                       const T &>::type)  // Safe ADL lookup of make_status_code(), returns void if not found
BOOST_OUTCOME_SYSTEM_ERROR2_TREQUIRES(BOOST_OUTCOME_SYSTEM_ERROR2_TPRED(is_status_code<MakeStatusCodeResult>::value))  // ADL makes a status code
inline bool operator==(const T &a, const errored_status_code<DomainType1> &b)
{
  return b.equivalent(make_status_code(a));
}
//! True if the status code's are not semantically equal via `equivalent()` to `make_status_code(T)`.
BOOST_OUTCOME_SYSTEM_ERROR2_TEMPLATE(class DomainType1, class T,  //
                       class MakeStatusCodeResult = typename detail::safe_get_make_status_code_result<
                       const T &>::type)  // Safe ADL lookup of make_status_code(), returns void if not found
BOOST_OUTCOME_SYSTEM_ERROR2_TREQUIRES(BOOST_OUTCOME_SYSTEM_ERROR2_TPRED(is_status_code<MakeStatusCodeResult>::value))  // ADL makes a status code
inline bool operator!=(const errored_status_code<DomainType1> &a, const T &b)
{
  return !a.equivalent(make_status_code(b));
}
//! True if the status code's are semantically equal via `equivalent()` to `make_status_code(T)`.
BOOST_OUTCOME_SYSTEM_ERROR2_TEMPLATE(class T, class DomainType1,  //
                       class MakeStatusCodeResult = typename detail::safe_get_make_status_code_result<
                       const T &>::type)  // Safe ADL lookup of make_status_code(), returns void if not found
BOOST_OUTCOME_SYSTEM_ERROR2_TREQUIRES(BOOST_OUTCOME_SYSTEM_ERROR2_TPRED(is_status_code<MakeStatusCodeResult>::value))  // ADL makes a status code
inline bool operator!=(const T &a, const errored_status_code<DomainType1> &b)
{
  return !b.equivalent(make_status_code(a));
}
//! True if the status code's are semantically equal via `equivalent()` to
//! `quick_status_code_from_enum<T>::code_type(b)`.
template <class DomainType1, class T,                                                     //
          class QuickStatusCodeType = typename quick_status_code_from_enum<T>::code_type  // Enumeration has been
                                                                                          // activated
          >
inline bool operator==(const errored_status_code<DomainType1> &a, const T &b)
{
  return a.equivalent(QuickStatusCodeType(b));
}
//! True if the status code's are semantically equal via `equivalent()` to
//! `quick_status_code_from_enum<T>::code_type(a)`.
template <class T, class DomainType1,                                                     //
          class QuickStatusCodeType = typename quick_status_code_from_enum<T>::code_type  // Enumeration has been
                                                                                          // activated
          >
inline bool operator==(const T &a, const errored_status_code<DomainType1> &b)
{
  return b.equivalent(QuickStatusCodeType(a));
}
//! True if the status code's are not semantically equal via `equivalent()` to
//! `quick_status_code_from_enum<T>::code_type(b)`.
template <class DomainType1, class T,                                                     //
          class QuickStatusCodeType = typename quick_status_code_from_enum<T>::code_type  // Enumeration has been
                                                                                          // activated
          >
inline bool operator!=(const errored_status_code<DomainType1> &a, const T &b)
{
  return !a.equivalent(QuickStatusCodeType(b));
}
//! True if the status code's are not semantically equal via `equivalent()` to
//! `quick_status_code_from_enum<T>::code_type(a)`.
template <class T, class DomainType1,                                                     //
          class QuickStatusCodeType = typename quick_status_code_from_enum<T>::code_type  // Enumeration has been
                                                                                          // activated
          >
inline bool operator!=(const T &a, const errored_status_code<DomainType1> &b)
{
  return !b.equivalent(QuickStatusCodeType(a));
}

BOOST_OUTCOME_SYSTEM_ERROR2_NAMESPACE_END

#endif
